"use client";
import Link from "next/link";
import { useEffect, useState } from "react";
import { useSession } from "next-auth/react";
import { useRouter } from "next/router";
import { useLanguage } from "@/contexts/LanguageContext";
import {
  Plus,
  RefreshCw,
  AlertTriangle,
  Search,
  Filter,
  X,
} from "lucide-react";

type TicketRow = {
  id: string;
  title: string;
  severity: number;
  department: string;
  createdAt: string;
  closedAt?: string | null;
  lastKnownStatus?: number | null;
  createdBy?: {
    id: string;
    email: string;
    name?: string;
    role: string;
  } | null;
  anonymousEmail?: string | null;
  anonymousName?: string | null;
};

export default function TicketsList() {
  const { data: session, status } = useSession();
  const router = useRouter();
  const { t } = useLanguage();
  const [rows, setRows] = useState<TicketRow[]>([]);
  const [departments, setDepartments] = useState<string[]>([]);
  const [loading, setLoading] = useState(true);
  const [err, setErr] = useState<string | null>(null);

  // Filters
  const [filters, setFilters] = useState({
    status: "",
    severity: "",
    department: "",
    search: "",
  });
  const [showFilters, setShowFilters] = useState(false);

  const fetchTickets = () => {
    if (status !== "authenticated") return;
    setLoading(true);
    const params = new URLSearchParams();
    Object.entries(filters).forEach(([key, val]) => {
      if (val) params.append(key, val);
    });
    const query = params.toString();
    const url = query ? `/api/tickets?${query}` : "/api/tickets";
    fetch(url)
      .then((r) => r.json())
      .then((j) => {
        if (!j.ok) throw new Error(j.error || "Failed to load tickets");
        setRows(j.tickets);
      })
      .catch((e) => setErr(String(e)))
      .finally(() => setLoading(false));
  };

  const fetchDepartments = () => {
    if (status !== "authenticated") return;
    // Fetch all tickets without filters to get all departments
    fetch("/api/tickets")
      .then((r) => r.json())
      .then((j) => {
        if (j.ok) {
          const uniqueDepartments = [
            ...new Set(j.tickets.map((t: TicketRow) => t.department)),
          ] as string[];
          setDepartments(uniqueDepartments.sort());
        }
      })
      .catch((e) => console.error("Failed to fetch departments:", e));
  };

  useEffect(() => {
    fetchTickets();
  }, [status, filters]);

  useEffect(() => {
    fetchDepartments();
  }, [status]);

  // Auto-refresh every 30 seconds
  useEffect(() => {
    if (status !== "authenticated") return;

    const interval = setInterval(() => {
      fetchTickets();
    }, 30000); // 30 seconds

    return () => clearInterval(interval);
  }, [status, filters]);

  if (status === "unauthenticated") {
    return (
      <main className='flex flex-col items-center justify-center min-h-[60vh] text-center text-[color:var(--color-foreground)]'>
        <AlertTriangle className='w-10 h-10 mb-3 text-[color:var(--color-warning)]' />
        <p className='text-lg font-medium mb-4'>
          You need to sign in to view tickets.
        </p>
        <Link
          href='/login'
          className='btn-primary px-4 py-2 rounded-lg font-medium'
        >
          Go to Login
        </Link>
      </main>
    );
  }

  const clearFilters = () =>
    setFilters({ status: "", severity: "", department: "", search: "" });
  const hasActiveFilters = Object.values(filters).some((v) => v);

  return (
    <main className='w-full p-6 space-y-8'>
      {/* Header */}
      <div className='flex items-center justify-between flex-wrap gap-3'>
        <div>
          <h1 className='text-3xl font-bold text-[color:var(--color-foreground)]'>
            {t("tickets.overview")}
          </h1>
          <p className='text-[color:var(--color-muted)]'>
            {t("tickets.description")}
          </p>
        </div>

        <div className='flex items-center gap-3'>
          <button
            onClick={() => setShowFilters(!showFilters)}
            className={`flex items-center gap-2 px-4 py-2 rounded-md text-sm font-medium transition ${
              showFilters || hasActiveFilters
                ? "bg-[color:var(--color-primary)] text-white"
                : "border border-[color:var(--color-border)] hover:bg-[color:var(--color-muted-bg)]"
            }`}
          >
            <Filter className='w-4 h-4' />
            {t("tickets.filters")}
            {hasActiveFilters && (
              <span className='text-xs ml-1'>
                ({Object.values(filters).filter(Boolean).length})
              </span>
            )}
          </button>
          <button
            onClick={fetchTickets}
            className='flex items-center gap-2 px-4 py-2 border border-[color:var(--color-border)] rounded-md text-sm font-medium hover:bg-[color:var(--color-muted-bg)]'
          >
            <RefreshCw className='w-4 h-4' />
            {t("common.refresh")}
          </button>
          <Link
            href='/tickets/new'
            className='flex items-center gap-2 px-4 py-2 rounded-md text-sm font-medium bg-[color:var(--color-primary)]  text-white'
          >
            <Plus className='w-4 h-4' />
            {t("sidebar.newTicket")}
          </Link>
        </div>
      </div>

      {/* Filter Panel */}
      {showFilters && (
        <section className='p-5 bg-[color:var(--color-card-bg,white)] rounded-xl border border-[color:var(--color-border)] shadow-sm'>
          <div className='grid sm:grid-cols-2 lg:grid-cols-4 gap-4'>
            <FilterInput
              label={t("common.search")}
              icon={<Search className='w-4 h-4 text-gray-400' />}
              placeholder={t("tickets.searchPlaceholder")}
              value={filters.search}
              onChange={(v) => setFilters((p) => ({ ...p, search: v }))}
            />
            <FilterSelect
              label={t("tickets.statusLabel")}
              value={filters.status}
              onChange={(v) => setFilters((p) => ({ ...p, status: v }))}
              options={[
                ["", t("tickets.allStatus")],
                ["open", t("tickets.statusValues.open")],
                ["closed", t("tickets.statusValues.closed")],
              ]}
            />
            <FilterSelect
              label={t("tickets.severityLabel")}
              value={filters.severity}
              onChange={(v) => setFilters((p) => ({ ...p, severity: v }))}
              options={[
                ["", t("tickets.allSeverity")],
                ["1", "Severity 1"],
                ["2", "Severity 2"],
                ["3", "Severity 3"],
                ["4", "Severity 4"],
                ["5", "Severity 5"],
              ]}
            />
            <FilterSelect
              label={t("tickets.department") || "Department"}
              value={filters.department}
              onChange={(v) => setFilters((p) => ({ ...p, department: v }))}
              options={[
                ["", t("tickets.allDepartments") || "All Departments"],
                ...departments.map((dept) => [dept, dept] as [string, string]),
              ]}
            />
          </div>

          <div className='flex items-center justify-between mt-6 border-t border-[color:var(--color-border)] pt-4'>
            <p className='text-sm text-[color:var(--color-muted)]'>
              {hasActiveFilters
                ? `${rows.length} ${t("tickets.foundTickets") || "tickets found"}`
                : t("tickets.showingAllTickets") || "Showing all tickets"}
            </p>
            {hasActiveFilters && (
              <button
                onClick={clearFilters}
                className='flex items-center gap-1 text-sm text-[color:var(--color-muted)] hover:text-[color:var(--color-foreground)] transition'
              >
                <X className='w-4 h-4' /> {t("tickets.clearFilters")}
              </button>
            )}
          </div>
        </section>
      )}

      {/* Results */}
      {loading && (
        <div className='flex items-center gap-2 text-[color:var(--color-muted)]'>
          <RefreshCw className='w-4 h-4 animate-spin' />
          {t("tickets.loading")}
        </div>
      )}

      {err && (
        <div className='rounded-lg border-l-4 border-red-500 bg-red-50 p-4 text-red-700 flex items-center gap-2'>
          <AlertTriangle className='w-4 h-4' /> {err}
        </div>
      )}

      {!loading && !err && rows.length > 0 && (
        <div className='overflow-x-auto border border-[color:var(--color-border)] rounded-xl bg-[color:var(--color-card-bg,white)] shadow-sm'>
          <table className='w-full text-sm'>
            <thead className='bg-[color:var(--color-muted-bg)] text-[color:var(--color-foreground)]/70'>
              <tr>
                <th className='px-4 py-3 text-left font-semibold'>
                  {t("tickets.table.title")}
                </th>
                <th className='px-4 py-3 text-left font-semibold'>
                  {t("tickets.table.status")}
                </th>
                <th className='px-4 py-3 text-left font-semibold'>
                  {t("tickets.table.department")}
                </th>
                <th className='px-4 py-3 text-left font-semibold'>
                  {t("tickets.table.severity")}
                </th>
                <th className='px-4 py-3 text-left font-semibold'>
                  {t("tickets.table.created")}
                </th>
                <th className='px-4 py-3 text-left font-semibold'>
                  {t("tickets.table.reporter")}
                </th>
              </tr>
            </thead>
            <tbody>
              {rows.map((ticket) => (
                <tr
                  key={ticket.id}
                  className='border-t border-[color:var(--color-border)] hover:bg-[color:var(--color-muted-bg)]/60 transition cursor-pointer group'
                  onClick={() => router.push(`/tickets/${ticket.id}`)}
                >
                  <td className='px-4 py-3'>
                    <div className='text-[color:var(--color-primary)] font-medium group-hover:underline'>
                      {ticket.title}
                    </div>
                  </td>
                  <td className='px-4 py-3'>
                    <StatusBadge
                      lastKnownStatus={ticket.lastKnownStatus}
                      closedAt={ticket.closedAt}
                      t={t}
                    />
                  </td>
                  <td className='px-4 py-3'>{ticket.department}</td>
                  <td className='px-4 py-3'>
                    <SeverityBadge severity={ticket.severity} />
                  </td>
                  <td className='px-4 py-3 text-[color:var(--color-muted)]'>
                    {new Date(ticket.createdAt).toLocaleString()}
                  </td>
                  <td className='px-4 py-3 text-[color:var(--color-muted)]'>
                    {ticket.createdBy ? (
                      <>
                        <span className='font-medium text-[color:var(--color-foreground)]'>
                          {ticket.createdBy.name || ticket.createdBy.email}
                        </span>
                        <div className='text-xs uppercase'>
                          {ticket.createdBy.role}
                        </div>
                      </>
                    ) : ticket.anonymousEmail ? (
                      <>
                        <span className='font-medium'>
                          {ticket.anonymousName || ticket.anonymousEmail}
                        </span>
                        <div className='text-xs uppercase text-[color:var(--color-muted)]'>
                          {t("tickets.anonymous") || "Anonymous"}
                        </div>
                      </>
                    ) : (
                      "—"
                    )}
                  </td>
                </tr>
              ))}
            </tbody>
          </table>
        </div>
      )}

      {!loading && !err && rows.length === 0 && (
        <div className='text-center text-[color:var(--color-muted)] py-12 border border-dashed rounded-lg'>
          {t("tickets.noTickets")}
        </div>
      )}
    </main>
  );
}

/* ---------------- Components ---------------- */

function FilterInput({
  label,
  icon,
  placeholder,
  value,
  onChange,
}: {
  label: string;
  icon?: React.ReactNode;
  placeholder?: string;
  value: string;
  onChange: (v: string) => void;
}) {
  return (
    <div>
      <label className='block text-sm font-medium mb-2 text-[color:var(--color-foreground)]/80'>
        {label}
      </label>
      <div className='relative'>
        {icon && (
          <span className='absolute left-3 top-1/2 -translate-y-1/2'>
            {icon}
          </span>
        )}
        <input
          type='text'
          placeholder={placeholder}
          value={value}
          onChange={(e) => onChange(e.target.value)}
          className={`w-full ${
            icon ? "pl-10" : "pl-3"
          } pr-3 py-2 border border-[color:var(--color-border)] rounded-md text-sm focus:ring-2 focus:ring-[color:var(--color-primary)] outline-none`}
        />
      </div>
    </div>
  );
}

function FilterSelect({
  label,
  value,
  onChange,
  options,
}: {
  label: string;
  value: string;
  onChange: (v: string) => void;
  options: [string, string][];
}) {
  return (
    <div>
      <label className='block text-sm font-medium mb-2 text-[color:var(--color-foreground)]/80'>
        {label}
      </label>
      <select
        value={value}
        onChange={(e) => onChange(e.target.value)}
        className='w-full px-3 py-2 border border-[color:var(--color-border)] rounded-md text-sm focus:ring-2 focus:ring-[color:var(--color-primary)] outline-none'
      >
        {options.map(([val, text]) => (
          <option key={val} value={val}>
            {text}
          </option>
        ))}
      </select>
    </div>
  );
}

function StatusBadge({
  lastKnownStatus,
  closedAt,
  t,
}: {
  lastKnownStatus?: number | null;
  closedAt?: string | null;
  t: (key: string) => string;
}) {
  // Determine status from lastKnownStatus if available, otherwise fallback to closedAt
  const status =
    lastKnownStatus !== null && lastKnownStatus !== undefined
      ? lastKnownStatus
      : closedAt
      ? 3
      : 0; // 0 = Open, 3 = Closed

  const getStatusText = (status: number) => {
    const statusMap = ["open", "inProgress", "resolved", "closed", "canceled"];
    return t(`tickets.statusValues.${statusMap[status] || "open"}`);
  };

  const colors = [
    "bg-blue-100 text-blue-700",
    "bg-yellow-100 text-yellow-700",
    "bg-green-100 text-green-700",
    "bg-gray-100 text-gray-700",
    "bg-red-100 text-red-700",
  ];

  const config = {
    text: getStatusText(status),
    colors: colors[status] || colors[0],
  };

  return (
    <span
      className={`inline-flex items-center gap-1 px-2 py-0.5 rounded-full text-xs font-medium ${config.colors}`}
    >
      {config.text}
    </span>
  );
}

function SeverityBadge({ severity }: { severity: number }) {
  const colors =
    severity >= 4
      ? "bg-red-100 text-red-600"
      : severity === 3
      ? "bg-yellow-100 text-yellow-700"
      : "bg-green-100 text-green-700";
  return (
    <span
      className={`inline-flex items-center gap-1 px-2 py-0.5 rounded-full text-xs font-medium ${colors}`}
    >
      {severity}
    </span>
  );
}
